/* timer.c
 *
 * 2005 Thomas Kulessa <tomk@khm.de>
 *
 */

#include <avr/io.h>
#include <avr/signal.h>
#include <avr/interrupt.h>

#include "global.h"
#include "timer.h"

#define MAX_TIMER_FUNC 2

typedef void (*ir_func)();

static ir_func timer0_functions[MAX_TIMER_FUNC];
static ir_func timer1_functions[MAX_TIMER_FUNC];


INTERRUPT(SIG_OVERFLOW0)
{
  unsigned char c = 0;
  while (c<MAX_TIMER_FUNC && timer0_functions[c] != 0) {
    (*timer0_functions[c++])();
  }
  
}

INTERRUPT(SIG_OVERFLOW1)
{
  unsigned char c = 0;
  while (c<MAX_TIMER_FUNC && timer1_functions[c] != 0) {
    (*timer1_functions[c++])();
  }
}

char timer0_init()
{
  unsigned char c;

  for (c=0;c<MAX_TIMER_FUNC;c++)
    timer0_functions[c] = 0;

  /* Reset counter */
  TCNT0 = 0;

  /* Clock set to 16Mhz/8*/
  bit_clear(TCCR0, CS02); 
  bit_set(TCCR0, CS01); 
  bit_clear(TCCR0, CS00);

  /* Turn on timer0 interrupt and interrupts in general */
  timer_enable_int(_BV(TOIE0)); 
  sei();

  return 1;
}

char timer0_add_function(void (*ir)())
{
  unsigned char c = 0;
  while (c<MAX_TIMER_FUNC && timer0_functions[c]!=0) {c++;}
  if (c<MAX_TIMER_FUNC && timer0_functions[c] == 0) {
    timer0_functions[c] = ir;
    return 1;
  } else 
    return 0;
}

char timer1_init() 
{
  unsigned char c;

  for (c=0;c<MAX_TIMER_FUNC;c++)
    timer1_functions[c] = 0;

   /* Reset counter */
  TCNT1 = 0;

  /* Clock set to 16Mhz (ext.speed) */
  bit_clear(TCCR1B, CS02); 
  bit_clear(TCCR1B, CS01); 
  bit_set(TCCR1B, CS00);

  /* Turn on timer0 interrupt and interrupts in general */
  timer_enable_int(_BV(TOIE1)); 
  sei();

  return 1;
}

char timer1_add_function(void (*ir)())
{
  unsigned char c = 0;
  while (c<MAX_TIMER_FUNC && timer1_functions[c]!=0) {c++;}

  if (c<MAX_TIMER_FUNC && timer1_functions[c] == 0) {
    timer1_functions[c] = ir;
    return 1;
  } else 
    return 0;
}
