/*  
 * System tries to reach an equilibrium between sensor on "front" and "back".
 * Sensor setup in relation to motor positions:
 *  
 *
 * LDR0 IR0 SONIC0 SONIC1 IR1 LDR1
 *                ^ 
 *        MOTOR1  |  MOTOR0
 *                v
 * LDR2 IR2 SONIC2 SONIC3 IR3 LDR3
 *
 *
 * 2006 Thomas Kulessa <tomk@khm.de>
 *
 */

#include <avr/io.h>
#include <stdlib.h>
#include <math.h>
#include "actor.h"
#include "serial.h"
#include "global.h"
#include "timer.h"
#include "sonic.h"


#define MIN_SPEED 25

/* Numbers of different sensor types */
#define SONIC_NR 4
#define IR_NR 4
#define LDR_NR 4
#define MOTOR_NR 2


sonic_t sonic[SONIC_NR];
led_t led;
port_t port_led, port_sonic, port_motor;
motor_t motor[MOTOR_NR];

void init_controller();
void calibrate_sensors();
void motor_func0();
void motor_func1();


int main() 
{
  init_controller();
  calibrate_sensors();

  while( 1 ) {
    unsigned char c;
    float ir_val[IR_NR], sonic_val[SONIC_NR];
    float ir_impuls[MOTOR_NR], sonic_impuls[MOTOR_NR];
    float ir_diff, sonic_diff;

    /* Read sensor values */
    /* Value of 1 means minimal proximity. */
    for (c=0; c<IR_NR; c++) ir_val[c] = adc_get_norm(c);
#ifdef VEHICLE_1
    ir_val[3] = ir_val[2];
#endif

    /* Value of 1 means minimal proximity. 1-... */
    for (c=0; c<SONIC_NR; c++) sonic_val[c] = 1 - sonic_get_norm(&sonic[c]);

    /* Check if "front" or "back" stimulus is smaller and set speed. 
     * Vehicle gets the impuls to move in direction with smaller stimulus.
     * Sensors and motors have a negative feedback relation:
     * The bigger a sensor stimululs gets the smaller the motor impuls is,
     * with the result, that the vehicle should not bump into obstacles.
     */

    /* Same for infrared */
    ir_impuls[0] = -ir_val[1]+ir_val[3];
    ir_impuls[1] = -ir_val[0]+ir_val[2];

    /* And for sonic sensors */
    sonic_impuls[0] = sonic_val[2]-sonic_val[0];
    sonic_impuls[1] = sonic_val[3]-sonic_val[1];
    
    /* Set the speed finally, by simply adding the values of the 3 
     * sensor groups  
     */
    for (c=0; c<MOTOR_NR;c++) {
      float sum = 10*sonic_impuls[c]+ir_impuls[c];
      if (sum > 1) sum = 1;
      motor_set_speed(&motor[c], sum * MAX_SPEED);
    }
    delay_s(6);
    led_blink(&led, 1);
  }
}

void motor_func(motor_t *motor)
{
  if (!motor->speed == 0) {
    motor->cnt++;
    if (motor->is_on) {
      motor->max = motor->speed;
      if (motor->cnt>motor->max) {
	bit_clear(*motor->port->port, motor->pin_enable);
	motor->is_on=0;
	motor->cnt=0;
      }
    } else {
      motor->max = MAX_SPEED-motor->speed;
      if (motor->cnt>motor->max) {
	bit_set(*motor->port->port, motor->pin_enable);
	motor->is_on=1;
	motor->cnt=0;
      }
    }
  }
}

void motor_func0()
{
  motor_func(&(motor[0]));
}

void motor_func1()
{
  motor_func(&(motor[1]));
}


void init_controller()
{
  unsigned int i;


  usart_open();


   /* Init LED */
  port_init(&PORTB, &port_led);
  if (!led_init(&port_led, PORTB0, &led)) {
    fprintf(stderr, "Failed to init LED port.\n");
    exit(1);
  }
  led_blink(&led, 3);
  
  /* Init infrared and light sensors */
  if (IR_NR+LDR_NR>ADC_NR) {
    fprintf(stderr, "To many infrared and light sensors defined: %d.\n",
            IR_NR+LDR_NR);
    exit(0);
  }
  adc_init();


  /* Init sonic sensors */
  port_init(&PORTC, &port_sonic);
  for (i=0; i < SONIC_NR; i++) {
    if (!sonic_init(&port_sonic, i, &sonic[i])) {
      fprintf(stderr, "Failed to init sonic sensor %d.\n", i);
      exit(1);
    }
  }


  /* Init motors */
  port_init(&PORTD, &port_motor);

  if (!motor_init(&port_motor, PORTD5, PORTD6, PORTD4, &(motor[1])) ||
      !motor_init(&port_motor, PORTD2, PORTD3, PORTD7, &(motor[0]))) {
    fprintf(stderr, "Failed to init one of the motors.\n");
    exit(1);
  }
  timer0_init();
  timer0_add_function(&motor_func0);
  timer0_add_function(&motor_func1);

}


void calibrate_sensors()
{
  unsigned char c;
  /* Calibration values from sensor-test/calibration program */
#ifdef VEHICLE_1
  /* Vehicle 1 */
  const unsigned int ir_min[IR_NR]           = {   30,     0,     3,     1};
  const unsigned int ir_max[IR_NR]           = {  531,   527,   120,   503}; 
  const unsigned int ldr_min[LDR_NR] = {0,    5,    7,   8};
  const unsigned int ldr_max[LDR_NR] = {543,  432,  120, 78};
  const unsigned int sonic_min[SONIC_NR]     = {   75,    62,    44,    44};
  const unsigned int sonic_max[SONIC_NR]     = {19709, 16329, 14106, 12416};

  printf("Calibrating sensors for vehicle 1...\n");
#endif
#ifdef VEHICLE_2
  /* Vehicle 2 */
  const unsigned int ir_min[IR_NR]           = {    1,     2,     2,     3};
  const unsigned int ir_max[IR_NR]           = {  495,   504,   516,   508}; 
  const unsigned int ldr_min[LDR_NR] = {0,    5,    7,   8};
  const unsigned int ldr_max[LDR_NR] = {543,  432,  120, 78};
  const unsigned int sonic_min[SONIC_NR]     = {  102,    85,    57,    61};
  const unsigned int sonic_max[SONIC_NR]     = {19695, 16403, 14155, 12319};
  printf("Calibrating sensors for vehicle 2...\n");
#endif
#ifdef VEHICLE_3
  /* Vehicle 3 */
  const unsigned int ir_min[IR_NR]           = {    0,     0,     2,     0};
  const unsigned int ir_max[IR_NR]           = {  500,   517,   551,   505}; 
  const unsigned int ldr_min[LDR_NR] = {0,    5,    7,   8};
  const unsigned int ldr_max[LDR_NR] = {543,  432,  120, 78};
  const unsigned int sonic_min[SONIC_NR]     = {   75,    82,    51,    72};
  const unsigned int sonic_max[SONIC_NR]     = {19837, 16436, 14085, 12300};
  printf("Calibrating sensors for vehicle 3...\n");
#endif

  for (c=0; c<IR_NR; c++) adc_set_range(c, ir_min[c], ir_max[c]);
  for (c=IR_NR; c<IR_NR+LDR_NR; c++) adc_set_range(c, ldr_min[c], ldr_max[c]);
  for (c=0; c<SONIC_NR; c++) 
    sonic_set_range(&sonic[c], sonic_min[c], sonic_max[c]);
  printf("Done.\n");
}
