/*
 * Atmel Mega8 helper functions.
 *
 * - Access to Analog digital converter.
 * - functions for setting and clearing register bits.
 *
 * 2005 Thomas Kulessa <tomk@khm.de>
 *
 */

#ifndef __GLOBAL__
#define __GLOBAL__

#include <avr/io.h>

#define ADC_NR 8

/* Input/output port object (PORT A, ..., PORT D) */ 
typedef struct 
{
  volatile uint8_t *port;
  volatile uint8_t *ddr;
  volatile uint8_t *pin;
} port_t;




/* Halts program execution for a number of seconds. The processor is
   busy during this time doing nothing 
*/
void delay_s(unsigned char s);



/* Initialised a port object. First parameter points to a port register.
   All pins are set to input. Pull-off resitor is off.
   Second parameter is the port structure to be filled.
   Function returns 0 if initialisation fails.
*/
char port_init(volatile uint8_t*, port_t *);

/* Sets alls pins on port to a bit mask */
/* void port_set_directions(port_t*, unsigned char); */

/* Sets one pin on a port to a bit value */
void port_set_direction(port_t* port, unsigned char pin, unsigned char value);

void port_set(port_t*, unsigned char);

/* Sets a register bit to 1 */
#define bit_set(sfr, bit) (_SFR_BYTE(sfr) |= _BV(bit))


/* Sets a register bit to 0 */
#define bit_clear(sfr, bit) (_SFR_BYTE(sfr) &= ~_BV(bit))


/* Toggle a register bit. If it was o it gets 1 and vice versa. */
#define bit_toggle(sfr, bit) (_SFR_BYTE(sfr) ^= _BV(bit))


/* Initializes, starts and sets some defaults for the ADC converter.
   All previous of register ADCSRA and ADMUX will be erased.
   see page 203-205 of Atmel ATmega8 guide.
*/
void adc_init();
 
/* Returns the current value for a given ADC. Value has to be between 0-7.
   If value is greater 7, -1 is returned.
 */
int adc_get(unsigned char nr);

/* Sets the value range (minimum/maximum values) of a specified ADC. */
char adc_set_range(unsigned char, unsigned int, unsigned int);

/* Returns the normalized value of an ADC. Which means the value
   is a floating point number between 0-1.
 */
float adc_get_norm(unsigned char nr);

#endif
