
#define F_CPU 16000000UL 
#include <avr/delay.h>
#include <avr/io.h>

#include <stdlib.h>
#include <stdint.h>

#include "global.h"
#include "serial.h"



static int adc_min[ADC_NR], adc_max[ADC_NR];


void adc_init()
{
  /* See Mega8 guide p. 9 and 202-204 */
  unsigned char i;

  for (i=0; i<ADC_NR; i++) {
    adc_min[i] = 0;
    adc_max[i] = 32767;
  }

  ADCSRA = 0;
  ADMUX = 0;


  /* Turns on ADC */
  ADCSRA |= 1 << ADEN;

  /* Sets free running mode */
/*   ADCSRA |= 1 << ADFR;  */

  /* Start conversion */
/*   ADCSRA |= 1 << ADSC;  */

  /* Turn on interrupts globally */
/*   sei(); */

  /* Turn on ADC Conversion interupt */
/*   ADCSRA |= 1 << ADIE; */

  /* Select prescaler to division factor 128, so clock will run at 125khz*/
  ADCSRA |= (1<<ADPS2)|(1<<ADPS1)|(1<<ADPS0); 

  /* ARF, AV_CC with external capacitor at AREF pin (5V on KHM board )*/
  ADMUX |= (1<<REFS0);
}


int adc_get(unsigned char nr)
{
  if (nr<ADC_NR) {
    unsigned int val;
    ADMUX &= 0xF0; /* Clear channel bits. (0xF0=11110000) */
    ADMUX |= nr;   /* Set channel */
    ADCSRA |= (1<<ADSC); /* Start conversion */    
    loop_until_bit_is_set(ADCSRA, ADIF);
    val = ADCL | (ADCH<<8); /* Add high and low byte*/
    ADCSRA |= (1<<ADIF);
    return val;
  } else
    return -1;
}

char port_init(volatile uint8_t *port, port_t *portstr)
{
  if (port == 0 || portstr == 0)
    return 0;

  portstr->port = portstr->ddr = portstr->pin = 0;

  if (port == &PORTD) {
    portstr->port = port;
    portstr->ddr = &DDRD;
    portstr->pin = &PIND;
  } else if (port == &PORTB) {
    portstr->port = port;
    portstr->ddr = &DDRB;
    portstr->pin = &PINB;
  } else if (port == &PORTC) {
    portstr->port = port;
    portstr->ddr = &DDRC;
    portstr->pin = &PINC;
  } else if (port == &PORTA) {
    portstr->port = port;
    portstr->ddr = &DDRA;
    portstr->pin = &PINA;
  } else
    return 0;

  *(portstr->pin) = 0;
  *(portstr->ddr) = 0;
  *(portstr->port) = 0;

  return 1;
}

/* void port_set_directions(port_t* port, unsigned char dv) */
/* { */
/*   *(port->ddr) |= dv; /\* Falsch, wenn eins dann bleibt es so *\/ */

/*   *(port->ddr) = dv; */
/* } */

void port_set_direction(port_t* port, unsigned char pin, unsigned char value)
{
  bit_clear(*(port->ddr), pin);
  if (value>0)
    bit_set(*(port->ddr), pin);
}
void port_set(port_t* port, unsigned char pv)
{
  *(port->port) = pv;
}


void delay_s(unsigned char s) 
{
  unsigned char c;
  for (c=0; c<s*10; c++)
    _delay_ms(100);
}


float adc_get_norm(unsigned char nr)
{
  const float val = (float) adc_get(nr);

  if (val>=0) {
    return (val-adc_min[nr])/(adc_max[nr]-adc_min[nr]);
  } else 
    return -1;
}


char adc_set_range(unsigned char nr, unsigned int min, unsigned int max)
{
  if (nr<ADC_NR && min < 32768 && max < 32768 && max-min != 0) {
    adc_min[nr] = min;
    adc_max[nr] = max;
    return 1;
  } else 
    return 0;
}
