/*
 * Prints out IO and ADC min and max values. Used to test and
 * calibate the sensors.
 * 2006 Thomas Kulessa <tomk@khm.de>
 *
 */

#include <avr/io.h>
#include <stdlib.h>
#include "actor.h"
#include "serial.h"
#include "global.h"
#include "sonic.h"


#define SONIC_NR 4
#define IR_NR 4
#define LDR_NR 4

FILE* out;
unsigned int ir_min[IR_NR], ir_max[IR_NR], ldr_min[LDR_NR], ldr_max[LDR_NR];
long sonic_min[SONIC_NR], sonic_max[SONIC_NR];
sonic_t sonic[SONIC_NR];
led_t led;
port_t port_led, port_sonic;

void init_controller();

int main() 
{
  unsigned int i;
  
  init_controller();

  /* Main loop */
  while (1) {
    printf("IR:    ");
    for (i=0; i<IR_NR; i++) {
      const unsigned int val = adc_get(i);
      if (val < ir_min[i]) ir_min[i] = val;
      if (val > ir_max[i]) ir_max[i] = val;
      printf("(%5d, %5d, %5d) ", val, ir_min[i], ir_max[i]);      
    }

    printf("\nLDR:   ");
    for (i=0; i<LDR_NR; i++) {
      const unsigned int val = adc_get(IR_NR+i);
      if (val < ldr_min[i]) ldr_min[i] = val;
      if (val > ldr_max[i]) ldr_max[i] = val;
      printf("(%5d, %5d, %5d) ", val, ldr_min[i], ldr_max[i]);      
    }
    
    printf("\nSONIC: ");
    for (i=0; i<SONIC_NR; i++) {
      const long val = sonic_get(&sonic[i]);
      if (val != -1) {
	if (val < sonic_min[i]) sonic_min[i] = val;
	if (val > sonic_max[i]) sonic_max[i] = val;
      }
      printf("(%5ld, %5ld, %5ld) ", val, sonic_min[i], sonic_max[i]);
    }
    printf("\n \n");

    led_blink(&led, 1);
  }
  
  return 0;
}


void init_controller()
{
  unsigned int i;


  out = usart_open();

  fprintf(stderr, "\nStarting sensor test calibration...\n");

   /* Init LED */
  port_init(&PORTB, &port_led);
  if (!led_init(&port_led, PORTB0, &led)) {
    fprintf(stderr, "Failed to init LED port.\n");
    exit(1);
  }
  led_blink(&led, 3);
  
  /* Init infrared and light sensors */
  if (IR_NR+LDR_NR>ADC_NR) {
    fprintf(stderr, "To many infrared and light sensors defined: %d.\n",
            IR_NR+LDR_NR);
    exit(0);
  }
  for (i=0; i<IR_NR; i++) {
    ir_min[i] = 65535;
    ir_max[i] = 0;
  }
  for (i=0; i<LDR_NR; i++) {
    ldr_min[i] = 65535;
    ldr_max[i] = 0;
  }
  adc_init();


  /* Init sonic sensors */
  port_init(&PORTC, &port_sonic);
  for (i=0; i < SONIC_NR; i++) {
    if (!sonic_init(&port_sonic, i, &sonic[i])) {
      fprintf(stderr, "Failed to init sonic sensor %d.\n", i);
      exit(1);
    }
    sonic_min[i] = 65535;
    sonic_max[i] = 0;
  }
}
